import java.io.*;
import java.util.Hashtable;

public final class TurnList4 {
  private static final String[] masktxt = {
    "Ux", "Fx", "Lx", "ux", "dx", "fx", "bx", "lx", "rx", "Uux", "Ffx", "Llx", "udx", "fbx", "lrx", "UUx", "FFx", "LLx"
  };

  private static final int TNUM = Turn4.NUM;
  public short[] next;

  public TurnList4(int num, int mask) {
    System.err.print("Allowed turns:");
    for (int i = 0; i < masktxt.length; i++) {
      if ((1 << i & mask) != 0)
        System.err.print(" " + masktxt[i]);
    }
    System.err.println();
    System.err.print("Turn list table:");
    String name = "tl" + num + "x" + Integer.toOctalString(mask) + ".tab";
    if (loadTab(name)) {
      System.err.println(" Loaded");
      return;
    }
    System.err.println(" Generating...");
    int[] tab = new int[num * TNUM];
    fill(tab, new Transform4(), mask);
    System.err.println(" Optimizing...");
    // find out all active states
    boolean[] act = new boolean[num];
    act[0] = true;
    int all = 1;
    for (int i = 0; i < num; i++) {
      for (int t = 0; t < TNUM; t++) {
        if (tab[i * TNUM + t] != -1 && !act[tab[i * TNUM + t]]) {
          act[tab[i * TNUM + t]] = true;
          all++;
        }
      }
    }
    int[] ind = new int[num];
    int n = 1;
    while (n > 0) {
      n = 0;
      for (int i = 0; i < num; i++)
        ind[i] = i;
      int sel = 0;
      for (int i = 0; i < num - 1; i++) {
        if (act[i]) {
          sel++;
          System.err.print("  " + i + "  " + sel + "/" + all + "\t\t\r");
          for (int j = i + 1; j < num; j++) {
            if (act[j]) {
              boolean eq = true;
              for (int t = 0; t < TNUM; t++) {
                if (tab[i * TNUM + t] != tab[j * TNUM + t]) {
                  eq = false;
                  break;
                }
              }
              if (eq) { // equal state?
                ind[j] = i;
                if (act[j]) {
                  act[j] = false;
                  all--;
                }
                n++;
              }
            }
          }
        }
      }
      // reduce equal states
      for (int i = 0; i < num; i++) {
        if (act[i]) {
          for (int t = 0; t < TNUM; t++) {
            int x = tab[i * TNUM + t];
            if (x >= 0 && ind[x] != x)
              tab[i * TNUM + t] = ind[x];
          }
        }
      }
      System.err.println("  equal states: " + n + ", active states: " + all);
    }
    int d = 0;
    int dup = 0;
    int tot = 0;
    for (int i = 0; i < num; i++) {
      if (act[i]) {
        d++;
        for (int t = 0; t < TNUM; t++) {
          tot++;
          if (tab[i * TNUM + t] == -1)
            dup++;
        }
      }
    }
    System.err.println("  active states: " + d + "/" + num);
    System.err.println("  cut: " + dup + "/" + tot);
    System.err.println(" Reducing...");
    int st = 0;
    // get new numbers for states
    for (int i = 0; i < num; i++)
      ind[i] = act[i] ? st++ : -1;
    next = new short[d * TNUM];
    int j = 0;
    // renumber states
    for (int i = 0; i < num; i++) {
      if (act[i]) {
        for (int t = 0; t < TNUM; t++) {
          int x = tab[i * TNUM + t];
          if (x < 0)
            next[j * TNUM + t] = -1;
          else {
            next[j * TNUM + t] = (short)ind[x];
            if (ind[x] < 0)
              System.err.print(" !!");
          }
        }
        j++;
      }
    }
    if (j != d)
      System.err.println(" !!!");
    System.err.println(" Writing to '" + name + "'...");
    saveTab(name);
  }

  private static void fill(int[] tab, Transform4 transform, int mask) {
    int num = tab.length / TNUM;
    State[] states = new State[num];
    Hashtable hash = new Hashtable();
    // put the initial state
    states[0] = State.init();
    hash.put(states[0], new Integer(0));
    int dup = 0;
    int from = 0;
    int to = 1;
    // compute num unique states
    System.err.println("  Generating states...");
    while (from < to) {
      System.err.print("   " + from + "->" + to + "/" + num + "\t\t\t\r");
      for (int t = 0; t < TNUM; t++) {
        if ((mask & Turn4.turnMask[t]) == 0) {
          tab[from * TNUM + t] = -1;
          dup++;
        }
        else {
          State ns = states[from].turn(t, transform);
          Integer nm = (Integer)hash.get(ns);
          if (nm != null) {
            int val = nm.intValue();
            tab[from * TNUM + t] = -1;
            dup++;
          }
          else {
            states[to] = ns;
            hash.put(ns, new Integer(to));
            tab[from * TNUM + t] = to;
            to++;
            if (to >= num)
              break;
          }
        }
      }
      if (to >= num)
        break;
      from++;
    }
    System.err.println("   cut: " + dup + "/" + num * TNUM + "\t\t\t");
    System.err.println("  Searching duplicities...");
    while (from < to) {
      System.err.print("   " + from + "->" + to + "\t\t\t\r");
      for (int t = 0; t < TNUM; t++) {
        if ((mask & Turn4.turnMask[t]) == 0) {
          tab[from * TNUM + t] = -1;
          dup++;
        }
        else {
          if (tab[from * TNUM + t] == 0) {
            State ns = states[from].turn(t, transform);
            if (hash.get(ns) != null) {
              tab[from * TNUM + t] = -1;
              dup++;
            }
          }
        }
      }
      from++;
    }
    System.err.println("   cut: " + dup + "/" + num * TNUM + "\t\t\t");
    System.err.println("  Linking states...");
    for (from = 0; from < num; from++) {
      System.err.print("   " + from + "/" + num + "\t\t\t\r");
      for (int t = 0; t < TNUM; t++) {
        if (tab[from * TNUM + t] == 0) {
          State s = states[from];
          for (int m = 1; m < s.move.length - 1; m++) {
            State ns = states[0];
            for (int mt = m; mt < s.move.length; mt++)
              ns = ns.turn(s.move[mt], transform);
            ns = ns.turn(t, transform);
            Integer nm = (Integer)hash.get(ns);
            if (nm != null) {
              int val = nm.intValue();
              if (ns.sameMove(states[val]))
                tab[from * TNUM + t] = val;
              else {
                tab[from * TNUM + t] = -1;
                dup++;
              }
              break;
            }
          }
          if (tab[from * TNUM + t] == 0)
            System.err.print(" !?!");
        }
      }
    }
    System.err.println("   cut: " + dup + "/" + num * TNUM + "\t\t\t");
  }

  private void saveTab(String name) {
    try {
      OutputStream os = new BufferedOutputStream(new FileOutputStream(name));
      for (int i = 0; i < next.length; i++) {
        os.write(next[i] >> 8 & 255);
        os.write(next[i] & 255);
      }
      os.close();
    }
    catch (IOException ex) {
      System.err.println(" File writing error!");
    }
  }

  private boolean loadTab(String name) {
    try {
      File file = new File(name);
      int len = (int)file.length();
      if (len % (TNUM * 2) != 0)
        return false;
      InputStream is = new BufferedInputStream(new FileInputStream(file));
      next = new short[len / 2];
      for (int i = 0; i < len / 2; i++) {
        next[i] = (short)(is.read() << 8);
        next[i] |= is.read();
      }
      is.close();
      return true;
    }
    catch (IOException ex) {
      System.err.println(" File reading error!");
      return false;
    }
  }
}

final class State {
  private short cornTwist;
  private short cornPerm;
  private short midLocU;
  private short midLocD;
  private short midLocF;
  private short midLocB;
  private short midLocL;
  private short midLocR;
  private short edgePosUF;
  private short edgePosUR;
  private short edgePosUB;
  private short edgePosUL;
  private short edgePosDF;
  private short edgePosDR;
  private short edgePosDB;
  private short edgePosDL;
  private short edgePosFR;
  private short edgePosFL;
  private short edgePosBR;
  private short edgePosBL;
  public byte[] move;

  private State() {}

  public boolean equals(Object o) {
    if (!(o instanceof State))
      return false;
    State s = (State)o;
    if (cornTwist != s.cornTwist)
      return false;
    if (cornPerm != s.cornPerm)
      return false;
    if (midLocU != s.midLocU)
      return false;
    if (midLocD != s.midLocD)
      return false;
    if (midLocF != s.midLocF)
      return false;
    if (midLocB != s.midLocB)
      return false;
    if (midLocL != s.midLocL)
      return false;
    if (midLocR != s.midLocR)
      return false;
    if (edgePosUF != s.edgePosUF)
      return false;
    if (edgePosUR != s.edgePosUR)
      return false;
    if (edgePosUB != s.edgePosUB)
      return false;
    if (edgePosUL != s.edgePosUL)
      return false;
    if (edgePosDF != s.edgePosDF)
      return false;
    if (edgePosDR != s.edgePosDR)
      return false;
    if (edgePosDB != s.edgePosDB)
      return false;
    if (edgePosDL != s.edgePosDL)
      return false;
    if (edgePosFR != s.edgePosFR)
      return false;
    if (edgePosFL != s.edgePosFL)
      return false;
    if (edgePosBR != s.edgePosBR)
      return false;
    if (edgePosBL != s.edgePosBL)
      return false;
    return true;
  }

  public boolean sameMove(State s) {
    if (move.length != s.move.length)
      return false;
    for (int i = 0; i < move.length; i++)
      if (move[i] != s.move[i])
        return false;
    return true;
  }

  public State turn(int turn, Transform4 transform) {
    State state = new State();
    state.cornTwist = transform.cornTwist.turn[turn][cornTwist];
    state.cornPerm = transform.cornPerm.turn[turn][cornPerm];
    state.midLocU = transform.midLoc[0].turn[turn][midLocU];
    state.midLocD = transform.midLoc[1].turn[turn][midLocD];
    state.midLocF = transform.midLoc[2].turn[turn][midLocF];
    state.midLocB = transform.midLoc[3].turn[turn][midLocB];
    state.midLocL = transform.midLoc[4].turn[turn][midLocL];
    state.midLocR = transform.midLoc[5].turn[turn][midLocR];
    state.edgePosUF = transform.edgePos[0].turn[turn][edgePosUF];
    state.edgePosUR = transform.edgePos[1].turn[turn][edgePosUR];
    state.edgePosUB = transform.edgePos[2].turn[turn][edgePosUB];
    state.edgePosUL = transform.edgePos[3].turn[turn][edgePosUL];
    state.edgePosDF = transform.edgePos[4].turn[turn][edgePosDF];
    state.edgePosDR = transform.edgePos[5].turn[turn][edgePosDR];
    state.edgePosDB = transform.edgePos[6].turn[turn][edgePosDB];
    state.edgePosDL = transform.edgePos[7].turn[turn][edgePosDL];
    state.edgePosFR = transform.edgePos[8].turn[turn][edgePosFR];
    state.edgePosFL = transform.edgePos[9].turn[turn][edgePosFL];
    state.edgePosBR = transform.edgePos[10].turn[turn][edgePosBR];
    state.edgePosBL = transform.edgePos[11].turn[turn][edgePosBL];
    state.move = new byte[move.length + 1];
    for (int i = 0; i < move.length; i++)
      state.move[i] = move[i];
    state.move[move.length] = (byte)turn;
    return state;
  }

  public static State init() {
    PackCornTwist4 packCornTwist = new PackCornTwist4(0);
    PackCornPerm4 packCornPerm = new PackCornPerm4(0);
    PackMidLoc4 packMidLocU = new PackMidLoc4(0, 0);
    PackMidLoc4 packMidLocD = new PackMidLoc4(0, 4);
    PackMidLoc4 packMidLocF = new PackMidLoc4(0, 8);
    PackMidLoc4 packMidLocB = new PackMidLoc4(0, 12);
    PackMidLoc4 packMidLocL = new PackMidLoc4(0, 16);
    PackMidLoc4 packMidLocR = new PackMidLoc4(0, 20);
    PackEdgePos4 packEdgePosUF = new PackEdgePos4(0, 0);
    PackEdgePos4 packEdgePosUR = new PackEdgePos4(0, 2);
    PackEdgePos4 packEdgePosUB = new PackEdgePos4(0, 4);
    PackEdgePos4 packEdgePosUL = new PackEdgePos4(0, 6);
    PackEdgePos4 packEdgePosDF = new PackEdgePos4(0, 8);
    PackEdgePos4 packEdgePosDR = new PackEdgePos4(0, 10);
    PackEdgePos4 packEdgePosDB = new PackEdgePos4(0, 12);
    PackEdgePos4 packEdgePosDL = new PackEdgePos4(0, 14);
    PackEdgePos4 packEdgePosFR = new PackEdgePos4(0, 16);
    PackEdgePos4 packEdgePosFL = new PackEdgePos4(0, 18);
    PackEdgePos4 packEdgePosBR = new PackEdgePos4(0, 20);
    PackEdgePos4 packEdgePosBL = new PackEdgePos4(0, 22);
    int len = packCornTwist.startLen();
    len += packCornPerm.startLen();
    len += packMidLocU.startLen();
    len += packMidLocD.startLen();
    len += packMidLocF.startLen();
    len += packMidLocB.startLen();
    len += packMidLocL.startLen();
    len += packMidLocR.startLen();
    len += packEdgePosUF.startLen();
    len += packEdgePosUR.startLen();
    len += packEdgePosUB.startLen();
    len += packEdgePosUL.startLen();
    len += packEdgePosDF.startLen();
    len += packEdgePosDR.startLen();
    len += packEdgePosDB.startLen();
    len += packEdgePosDL.startLen();
    len += packEdgePosFR.startLen();
    len += packEdgePosFL.startLen();
    len += packEdgePosBR.startLen();
    len += packEdgePosBL.startLen();
    if (len != 20)
      System.err.println("  Too many starting positions!");
    State state = new State();
    state.cornTwist = (short)packCornTwist.start(0);
    state.cornPerm = (short)packCornPerm.start(0);
    state.midLocU = (short)packMidLocU.start(0);
    state.midLocD = (short)packMidLocD.start(0);
    state.midLocF = (short)packMidLocF.start(0);
    state.midLocB = (short)packMidLocB.start(0);
    state.midLocL = (short)packMidLocL.start(0);
    state.midLocR = (short)packMidLocR.start(0);
    state.edgePosUF = (short)packEdgePosUF.start(0);
    state.edgePosUR = (short)packEdgePosUR.start(0);
    state.edgePosUB = (short)packEdgePosUB.start(0);
    state.edgePosUL = (short)packEdgePosUL.start(0);
    state.edgePosDF = (short)packEdgePosDF.start(0);
    state.edgePosDR = (short)packEdgePosDR.start(0);
    state.edgePosDB = (short)packEdgePosDB.start(0);
    state.edgePosDL = (short)packEdgePosDL.start(0);
    state.edgePosFR = (short)packEdgePosFR.start(0);
    state.edgePosFL = (short)packEdgePosFL.start(0);
    state.edgePosBR = (short)packEdgePosBR.start(0);
    state.edgePosBL = (short)packEdgePosBL.start(0);
    state.move = new byte[0];
    return state;
  }

  public String toString() {
    String s = "";
    for (int i = 0; i < move.length; i++)
      s += (i > 0 ? " " : "") + Solution4.turnSymbol[move[i]];
    return s;
  }

  public int hashCode() {
    int h = 31;
    h = h * 113 + cornTwist;
    h = h * 113 + cornPerm;
    h = h * 113 + midLocU;
    h = h * 113 + midLocD;
    h = h * 113 + midLocF;
    h = h * 113 + midLocB;
    h = h * 113 + midLocL;
    h = h * 113 + midLocR;
    h = h * 113 + edgePosUF;
    h = h * 113 + edgePosUR;
    h = h * 113 + edgePosUB;
    h = h * 113 + edgePosUL;
    h = h * 113 + edgePosDF;
    h = h * 113 + edgePosDR;
    h = h * 113 + edgePosDB;
    h = h * 113 + edgePosDL;
    h = h * 113 + edgePosFR;
    h = h * 113 + edgePosFL;
    h = h * 113 + edgePosBR;
    h = h * 113 + edgePosBL;
    return h;
  }
}
