#include <ext/hash_set>
#include <stdint.h>
#include <stdlib.h>
#include <vector>

#include "fill_rotate.h"
#include "main.h"
#include "input.h"
#include "prune.h"
#include "output.h"
#include "good_rotate.h"
#include "pack.h"

/* ############################################################################
 * Globals
 */

unsigned short rotate_corner_perm[CORNER_PERM_MAX_PACK][ROTATE_NUMBER];
unsigned short rotate_corner_orient[CORNER_ORIENT_MAX_PACK][ROTATE_NUMBER];
unsigned short rotate_center[CENTER_MAX_PACK][ROTATE_NUMBER];
unsigned short rotate_edge[EDGE_MAX_PACK][ROTATE_NUMBER];

unsigned char prune_corner[CORNER_PERM_MAX_PACK][CORNER_ORIENT_MAX_PACK];

#if POWERFULL_PRUNE > 0

unsigned char prune_center[CENTER_PACK_NUMBER][CENTER_MAX_PACK];
unsigned char prune_edge[EDGE_PACK_NUMBER][EDGE_MAX_PACK];

#endif

int allowed_moves[ROTATE_NUMBER];
char next_good_rotate[ROTATE_NUMBER][ROTATE_NUMBER+1];

int tree_i = 1;


/* ############################################################################
 * Distance
 */

inline int max(int a,int b)
{
	return (a>b?a:b);
}

#if POWERFULL_PRUNE == 2

inline unsigned char distance(state s)
{
	int m, i;
	m = prune_corner[s.corner_perm][s.corner_orient];
	
	for (i = 0; i < CENTER_PACK_NUMBER; i++)
		m = max(m, prune_center[i][s.center[i]]);

	for (i = 0; i < EDGE_PACK_NUMBER; i++)
		m = max(m, prune_edge[i][s.edge[i]]);

	return m;
}

#elif POWERFULL_PRUNE == 1

inline unsigned char distance(state *s)
{
return max(max(prune_corner[s->corner_perm][s->corner_orient],prune_edge[5][s->edge[5]]), max(prune_edge[0][s->edge[0]], prune_center[0][s->center[0]]));
}

#else

inline unsigned char distance(state *s)
{
return (prune_corner[s->corner_perm][s->corner_orient]);
}

#endif

/* ########################################################################### 
 * Solved state
 */

state solved()
{
	state s;
	int i;

	for (i = 0; i < CENTER_PACK_NUMBER; i++)
		s.center[i] = CENTER_SOLVED(i);

	for (i = 0; i < EDGE_PACK_NUMBER; i++)
		s.edge[i] = EDGE_SOLVED(i);

	s.corner_perm = CORNER_PERM_SOLVED;
	s.corner_orient = CORNER_ORIENT_SOLVED;

	s.moves = 0;
	return s;
}

/* ############################################################################
 * Initialisation
 */

void init()
{
	int i;

	fill_rotate_corner_orient(rotate_corner_orient);
	fill_rotate_corner_perm(rotate_corner_perm);
	fill_rotate_center(rotate_center);
	fill_rotate_edge(rotate_edge);

	#if POWERFULL_PRUNE > 0

	for (i = 0; i < CENTER_PACK_NUMBER; i++)
		fill_prune_center(prune_center[i], CENTER_SOLVED(i), rotate_center);

	for (i = 0; i < EDGE_PACK_NUMBER; i++)
		fill_prune_edge(prune_edge[i], EDGE_SOLVED(i), rotate_edge);
	
	#endif

	fill_prune_corner(prune_corner, CORNER_PERM_SOLVED, CORNER_ORIENT_SOLVED, rotate_corner_perm, rotate_corner_orient);


}

/* ############################################################################
 * Rotation
 */

void rotate_state_tree(state *s, state *t, int rotation)
{
	int i;

	for (i = 0; i < EDGE_PACK_NUMBER; i++)
		t->edge[i] = rotate_edge[s->edge[i]][rotation];
	
	for (i = 0; i < CENTER_PACK_NUMBER; i++)
		t->center[i] = rotate_center[s->center[i]][rotation];

	t->corner_orient = rotate_corner_orient[s->corner_orient][rotation];
	t->corner_perm = rotate_corner_perm[s->corner_perm][rotation];

	t->moves = (int64_t) (s->moves * (int64_t) ROTATE_NUMBER + (int64_t)rotation);
}

/* ############################################################################
 * State functions
 */

/*
void copy_state(state s,state *t)
{
	int i;
	
	for (i = 0; i < EDGE_PACK_NUMBER; i++)
		t->edge[i] = s.edge[i];
	
	for (i = 0; i < CENTER_PACK_NUMBER; i++)
		t->center[i] = s.center[i];

	t->corner_orient = s.corner_orient;
	t->corner_perm = s.corner_perm;
	
	t->moves = s.moves;
}
*/

/* ############################################################################
 * Tree search functions
 */

void next_state_depth(state tree[])
{	
	int last_rotation = tree[tree_i].moves % ROTATE_NUMBER;

	tree_i++;
	rotate_state_tree(&tree[tree_i-1],&tree[tree_i],next_good_rotate[last_rotation][0]);
}

void next_state_breath(state tree[])
{
	int last_rotation = tree[tree_i].moves % ROTATE_NUMBER;
	int gr = next_good_rotate[(tree[tree_i - 1].moves % ROTATE_NUMBER)][last_rotation+1];

	if (tree_i == 1)
   {
		last_rotation++;
		while (!allowed_moves[last_rotation] && last_rotation < ROTATE_NUMBER)
			last_rotation++;
		if (last_rotation < ROTATE_NUMBER)
		{
			rotate_state_tree(&tree[tree_i-1],&tree[tree_i],last_rotation);
		}
		else
		{
			tree_i--;
		}
	}
	else if (gr != -1)
	{
			rotate_state_tree(&tree[tree_i-1],&tree[tree_i],gr);
	}
	else
	{
		tree_i--;
		next_state_breath(tree);
	}
}

/* ############################################################################
 * Debug functions
 */

#if DEBUG_LEVEL > 1

void print_state(state s)
{
	int i;

	for (i = 0; i < EDGE_PACK_NUMBER; i++)
		printf("e%d=%d | ", i, s.edge[i]);

	for (i = 0; i < CENTER_PACK_NUMBER; i++)
		printf("c%d=%d | ", i, s.center[i]);

		printf("cp=%d | ", s.corner_perm);
		printf("co=%d\n", s.corner_orient);
}

#endif

#if DEBUG_LEVEL > 2

void print_rotation(state s, int moves_nbr)
{
	int i;
	int64_t moves = s.moves;

	for (i = 0; i < moves_nbr; i++)
	{
		print_rotate(moves % ROTATE_NUMBER);
		moves /= ROTATE_NUMBER;
	}
	printf("\n");
}

#endif

/* ############################################################################
 * Main loop
 */

int main()
{

	state t;
	__gnu_cxx::hash_set<state, state> dict;

	state * tree;
	tree = (state *)malloc((MAX_ROTATION + 3) * sizeof(state));

	vector<state> queue;
	int first = 0;
	int last = 1;
	int rot_number = 2;
	int first_rot;
	int last_rot;
	int point_n = 0;
	int i, m;
	int min;

	int current_limit;
	int64_t nodes;


	/* Initialisation */

	init();


	/* Filling queue solved */
	
	tree[0] = input(allowed_moves);
	fill_next_good_rotate(next_good_rotate,allowed_moves);
	
	#if DEBUG_LEVEL > 1
	printf("Scrambled state.\n"); 
	print_state(tree[0]);	
	#endif

	queue.push_back(solved());
	
	#if DEBUG_LEVEL > 1
	printf("Solved state.\n"); 
	print_state(queue[0]);	
	#endif
	
	#if DEBUG_LEVEL > 0
	printf("Starting filling around solved state.\n"); 
	
	printf("Depth 1"); 
	#endif
	for (m = 0; m < ROTATE_NUMBER; m++)
	{
		if (allowed_moves[m])
		{
			queue.push_back(state());
			rotate_state_tree(&queue[first],&queue[last],m);
			last++;
		}
	}

	last_rot = last;
	first++;

	#if DEBUG_LEVEL > 0
	printf(" .........cubes number %d to %d (%d).\n", first, last - 1, last - first); 
	printf("Depth %d ", 2); 
	#endif

	first_rot = first;
	while (rot_number < QUEUE_DEPTH)
	{

		#if DEBUG_LEVEL > 0
		if ( ((10 * (first - first_rot)) / (last_rot - first_rot)) > point_n  ){
			printf(".");
			fflush(stdout);
			point_n++;
		} 
		#endif

		for (m = 0; m < ROTATE_NUMBER; m++)
		{
			if (next_good_rotate[queue[first].moves % ROTATE_NUMBER][m] == m)
			{
				queue.push_back(state());
				rotate_state_tree(&queue[first], &queue[last], m);
				last++;
			}
		}
		first++;
		if (first == last_rot){
			last_rot = last;
			first_rot = first;
			point_n = 0;
			rot_number++;

			#if DEBUG_LEVEL > 0
			printf("cubes number %d to %d (%d).\n", first, last - 1, last - first); 
			printf("Depth %d ", rot_number); 
			#endif
		}
	}

	for (i = first; i < last; i++){

		#if DEBUG_LEVEL > 0
		if ( ((10 * (i - first_rot)) / (last - first_rot)) > point_n  ){
			printf(".");
			fflush(stdout);
			point_n++;
		} 
		#endif
		for (m = 0; m < ROTATE_NUMBER; m++)
			if (next_good_rotate[queue[i].moves % ROTATE_NUMBER][m] == m)
			{
				rotate_state_tree(&queue[i], &t,m);
				dict.insert(t);
			}
	}
	#if DEBUG_LEVEL > 0
	printf("finished.\n");
	#endif
	
	queue.clear();

	/* Main search */

	#if DEBUG_LEVEL > 0
	printf("Start searching !\n");
	#endif

	for (current_limit = QUEUE_DEPTH + 2; current_limit < MAX_ROTATION; current_limit++){

		tree_i = 1;

		for (i = 0; i < ROTATE_NUMBER; i++)
			{
				if (allowed_moves[i])
				{
					rotate_state_tree(&tree[0],&tree[1],i);
					break;
				}
			}

		nodes = 0;

		while (tree_i > 0)
		{
			
			#if DEBUG_LEVEL > 2
			print_rotation(tree[tree_i], tree_i);
			#endif

			#if DEBUG_LEVEL > 3
			print_state(tree[tree_i]);
			#endif


			nodes++;
			min = distance(&(tree[tree_i]));

			if (tree_i == current_limit - QUEUE_DEPTH)
				{
				if (min<= QUEUE_DEPTH)
					if (dict.find(tree[tree_i]) != dict.end())
						print_solution(tree[tree_i],(state)*(dict.find(tree[tree_i])), next_good_rotate, tree_i, QUEUE_DEPTH);
				next_state_breath(tree);
			}
			else if (min + tree_i > current_limit)
				next_state_breath(tree);
			else
				next_state_depth(tree);
		}
		#if DEBUG_LEVEL > 0
		printf("Depth %d finished. Nodes visited : %lld\n",current_limit,nodes);
		#endif
	}
}
