/*
 KSolve - Puzzle solving program.
 Copyright (C) 2007  Kre Krig

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include "move.h"
#include "rules.h"
#include "search.h"
#include "scramble.h"
#include "pruning.h"

#include <iostream>

#define PROFILE_K
#ifdef PROFILE_K
#include <windows.h>
#endif

using namespace std;

int main(int argc, char *argv[])
{
   clock_t start; 
   start = clock();

#ifdef PROFILE_K
   LARGE_INTEGER lFreq, lStart, lEnd;
   QueryPerformanceFrequency(&lFreq);
   QueryPerformanceCounter(&lStart);
#endif

   if(argc != 3){
      cerr << "Ksolve, v0.10. " << endl;
      cerr << "By: Kare Krig" << endl << endl;
      cerr << "Usage: ksolve [def-file] [scramble-file]" << endl;
      return EXIT_FAILURE;
   }

   // Load the puzzle rules
   rules ruleset(argv[1]);
   cout << "Ruleset loaded." << endl;

   map<string, dataset> datasets = ruleset.getDatasets();
   map<string, substate> solved = ruleset.getSolved();
   map<string, map<string, submove> > moves = ruleset.getMoves();
   vector<movepair> forbidden = ruleset.getForbiddenPairs();
   map<string, substate> ignore = ruleset.getIgnore();
   set<map<string, set<int> > > blocks = ruleset.getBlocks();


   // TEST
      set< pair<string, string> > forbidden_set;
      for(int i = 0; i < forbidden.size(); i++){
         pair<string, string> tmp( forbidden[i].a, forbidden[i].b );
         forbidden_set.insert( tmp );
      }

#ifdef PROFILE_K
   QueryPerformanceCounter(&lEnd);
   cout << "Pussel laddat: " << double(lEnd.QuadPart - lStart.QuadPart) / lFreq.QuadPart << endl;
   QueryPerformanceCounter(&lStart);
#endif

/*   map<string, map<string, submove> >::iterator moves_iter;
   map<string, submove>::iterator sub_iter;
   for(moves_iter = moves.begin(); moves_iter != moves.end(); moves_iter++){
      for(sub_iter = moves_iter->second.begin(); sub_iter != moves_iter->second.end(); sub_iter++){
         cout << sub_iter->second.size << endl;
      }
   }*/

   // Compute or load the pruning tables
   map<string, subprune> tables;
   string deffile(argv[1]);
   tables = getCompletePruneTables(solved, moves, datasets, ignore, deffile);

#ifdef PROFILE_K
   QueryPerformanceCounter(&lEnd);
   cout << "Tabeller berknade/laddade: " << double(lEnd.QuadPart - lStart.QuadPart) / lFreq.QuadPart << endl;
   QueryPerformanceCounter(&lStart);
#endif
   
   //datasets = updateDatasets(datasets, tables);
   updateDatasets(datasets, tables);

   // Load the scramble to be solved
   scramble states(argv[2], datasets);
   cout << "Scramble loaded." << endl;

   map<string, substate> state;
   state = states.getScramble();
   int max_depth = states.getMaxDepth();
   
#ifdef PROFILE_K
   QueryPerformanceCounter(&lEnd);
   cout << "Blandning laddad: " << double(lEnd.QuadPart - lStart.QuadPart) / lFreq.QuadPart << endl;
   QueryPerformanceCounter(&lStart);
#endif

   while( state.size() != 0 ){

      int depth = 0;
      string temp_a, temp_b;
      temp_a = " ";
      temp_b = ".";

      cout << endl << "Solving " << states.getName() << endl;
      cout << "Depth ";
 
   // The tree-search for the solution(s)
      while( !TreeSolve(state , solved, moves, datasets, tables, forbidden_set, ignore, blocks, depth, temp_a, temp_b) ){
         depth++;
         if( depth > max_depth ){
            cout << endl << "Max depth reached, aborting. " << endl;
            break;
         }
         cout << depth << " ";
      }
      cout << endl;

      state = states.getScramble();
      max_depth = states.getMaxDepth();
   }      

#ifdef PROFILE_K
   QueryPerformanceCounter(&lEnd);
   cout << "Lsning berknad: " << double(lEnd.QuadPart - lStart.QuadPart) / lFreq.QuadPart << endl;
   QueryPerformanceCounter(&lStart);
#endif

   cout << "Time: " << ( clock() - start ) / (double)CLOCKS_PER_SEC << "s" << endl;

   return EXIT_SUCCESS;
}
